// cpanel-api.js
// Base URL for the proxy endpoint
const proxyUrl = '/new-cpanel/proxy.php'; // Adjust if proxy.php is in a different directory

// Generic function to make API requests via the PHP proxy
async function makeApiRequest(endpoint, params = {}) {
  // Prepare the data to be sent to the proxy
  const requestData = {
    endpoint: endpoint,
    params: params
  };

  try {
    // Make the request to the PHP proxy
    const response = await fetch(proxyUrl, {
      method: 'POST', // Use POST method to send data to the proxy
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(requestData) // Send the request data as JSON
    });

    // Parse the JSON response from the proxy
    const data = await response.json();
    console.log('API Response:', data);
    return data;
  } catch (error) {
    console.error('Error making API request:', error);
    return null;
  }
}

// Function to check if a record already exists
async function recordExists(domain, type, value) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=parse_zone';
  const params = {
    zone: domain,
    type: type
  };
  const response = await makeApiRequest(endpoint, params);
  if (response && response.result && response.result.data) {
    return response.result.data.some(record => {
      const decodedData = record.data_b64 ? atob(record.data_b64[0]) : '';
      const decodedDname = record.dname_b64 ? atob(record.dname_b64) : '';
      return (decodedData.includes(value) || decodedDname.includes(value));
    });
  }
  return false;
}

// Function to list domains
async function listDomains() {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DomainInfo&cpanel_jsonapi_func=list_domains';
  return await makeApiRequest(endpoint);
}

// Function to list addon domains along with the main domain
async function listAddonDomains() {
  const domainData = await listDomains();
  if (domainData && domainData.result && domainData.result.data) {
    const addonDomains = domainData.result.data.addon_domains;
    const mainDomain = domainData.result.data.main_domain;

    // Create an array with the main domain and addon domains
    let domains = [mainDomain].concat(addonDomains);
    
    console.log('Domains:', domains);
    return domains;
  } else {
    console.error('Invalid domain data received from API.');
    return null;
  }
}

// Function to set CNAME record
async function setCnameRecord(domain, serial) {
  const exists = await recordExists(domain, 'CNAME', 'prox.itrackly.com.');
  if (exists) {
    console.log('CNAME record already exists. Skipping.');
    return { result: { data: { new_serial: serial } } };
  }
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=mass_edit_zone';
  const record = {
    dname: 'inst',
    ttl: 14400,
    record_type: 'CNAME',
    data: ['prox.itrackly.com.']
  };
  const params = {
    zone: domain,
    add: JSON.stringify(record),
    serial: serial
  };
  return await makeApiRequest(endpoint, params);
}

// Function to set additional CNAME record
async function setAdditionalCnameRecord(domain, serial) {
  const exists = await recordExists(domain, 'CNAME', 'open.sleadtrack.com.');
  if (exists) {
    console.log('Additional CNAME record already exists. Skipping.');
    return { result: { data: { new_serial: serial } } };
  }
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=mass_edit_zone';
  const record = {
    dname: 'smart',
    ttl: 14400,
    record_type: 'CNAME',
    data: ['open.sleadtrack.com.']
  };
  const params = {
    zone: domain,
    add: JSON.stringify(record),
    serial: serial
  };
  return await makeApiRequest(endpoint, params);
}

// Function to set DMARC record
async function setDmarcRecord(domain, serial) {
  const exists = await recordExists(domain, 'TXT', 'v=DMARC1;p=none;sp=none;adkim=s;aspf=s;pct=100;fo=1;rf=afrf;ri=86400');
  if (exists) {
    console.log('DMARC record already exists. Skipping.');
    return { result: { data: { new_serial: serial } } };
  }
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=mass_edit_zone';
  const record = {
    dname: `_dmarc`,
    ttl: 14400,
    record_type: 'TXT',
    data: ['v=DMARC1;p=none;sp=none;adkim=s;aspf=s;pct=100;fo=1;rf=afrf;ri=86400']
  };
  const params = {
    zone: domain,
    add: JSON.stringify(record),
    serial: serial
  };
  return await makeApiRequest(endpoint, params);
}

// Function to find line_index of TXT record to edit
async function findTxtRecordLineIndex(domain) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=parse_zone';
  const params = {
    zone: domain,
    type: 'TXT'
  };
  const response = await makeApiRequest(endpoint, params);
  if (response && response.result && response.result.data) {
    for (const record of response.result.data) {
      if (record.record_type === 'TXT') {
        const decodedDname = atob(record.dname_b64);
        if (decodedDname === `${domain}.`) {
          console.log(`Found matching record at line ${record.line_index}`);
          return record.line_index;
        }
      }
    }
  }
  console.log('No matching TXT record found for deletion.');
  return null;
}

// Function to delete TXT record
async function deleteTxtRecord(domain, lineIndex, serial) {
  if (lineIndex === null) {
    console.log('No matching TXT record found. Skipping deletion.');
    return { result: { data: { new_serial: serial } } };
  }
  console.log(`Attempting to delete TXT record at line ${lineIndex} for domain ${domain}`);
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=mass_edit_zone';
  const params = {
    zone: domain,
    serial: serial,
    remove: lineIndex
  };
  return await makeApiRequest(endpoint, params);
}

// Function to add new TXT record
async function addNewTxtRecord(domain, serial) {
  const exists = await recordExists(domain, 'TXT', 'v=spf1 +a +mx +ip4:165.140.70.70 include:relay.mailchannels.net ~all');
  if (exists) {
    console.log('TXT record already exists. Skipping.');
    return { result: { data: { new_serial: serial } } };
  }
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=mass_edit_zone';
  const record = {
    dname: `${domain}.`,
    ttl: 14400,
    record_type: 'TXT',
    data: ['v=spf1 +a +mx +ip4:165.140.70.70 include:relay.mailchannels.net ~all']
  };
  const params = {
    zone: domain,
    add: JSON.stringify(record),
    serial: serial
  };
  return await makeApiRequest(endpoint, params);
}

// Function to fetch DNS Serial Number
async function fetchDnsSerialNumber(domain) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=DNS&cpanel_jsonapi_func=parse_zone';
  const params = {
    zone: domain,
    type: 'SOA'
  };
  const response = await makeApiRequest(endpoint, params);
  if (response && response.result && response.result.data) {
    const soaRecord = response.result.data.find(record => record.record_type === 'SOA');
    if (soaRecord && soaRecord.data_b64) {
      const serialBase64 = soaRecord.data_b64[2];
      const serial = atob(serialBase64);
      return serial;
    }
  }
  throw new Error('Failed to fetch DNS serial number.');

  // Function to create an email account
async function createEmailAccount(email, password, domain, quota = 0, send_welcome_email = 1) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=Email&cpanel_jsonapi_func=add_pop';
  const params = {
    email: email,
    password: password,
    domain: domain,
    quota: quota,
    send_welcome_email: send_welcome_email
  };
  return await makeApiRequest(endpoint, params);
}
}

// Function to add email forwarder
async function addEmailForwarder(domain, email, fwdemail) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=Email&cpanel_jsonapi_func=add_forwarder';
  const params = {
    domain: domain,
    email: email,
    fwdopt: 'fwd',
    fwdemail: fwdemail
  };
  return await makeApiRequest(endpoint, params);
}

// Function to list email forwarders for a domain
async function listEmailForwarders(domain) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=Email&cpanel_jsonapi_func=list_forwarders';
  const params = {
    domain: domain
  };
  return await makeApiRequest(endpoint, params);
}

// Function to list email accounts
async function listEmailAccounts() {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=Email&cpanel_jsonapi_func=list_pops';
  const params = {
    no_validate: 1,
    skip_main: 0
  };
  return await makeApiRequest(endpoint, params);
}

// Function to get client settings for an email account
async function getClientSettings(email) {
  const endpoint = 'cpanel?cpanel_jsonapi_apiversion=3&cpanel_jsonapi_module=Email&cpanel_jsonapi_func=get_client_settings';
  const params = {
    email: email
  };
  return await makeApiRequest(endpoint, params);
}
